unit MailTemplateQuery;

{
  A database query for the mail template program.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - October 17 1999.
  Version 1.1 - April 25, 2000.
}

interface

uses
  Classes, Db, DbTables, MailTemplateConst;

type
  TMailTemplateQuery = class(TObject)
  private
    FDatabase: TDatabase;
    FFields: TStringList;
    FQuery: TQuery;
    function GetEOF: Boolean;
    procedure QueryAfterScroll(DataSet: TDataSet);
  public
    constructor Create(Props: TStrings; QuerySQL: string);
    destructor Destroy; override;
    property EOF: Boolean read GetEOF;
    property Fields: TStringList read FFields;
    procedure NextRecord;
  end;

implementation

{ TMailTemplateQuery ----------------------------------------------------------}

{ Initialisation - connect to database and open query }
constructor TMailTemplateQuery.Create(Props: TStrings; QuerySQL: string);
begin
  inherited Create;
  FFields   := TStringList.Create;
  FDatabase := TDatabase.Create(nil);
  with FDatabase do
  begin
    AliasName    := Props.Values[QueryAliasProp];
    DatabaseName := 'MailOut';
    LoginPrompt  := False;
    if Props.Values[QueryUserProp] <> '' then
      Params.Add('username=' + Props.Values[QueryUserProp]);
    if Props.Values[QueryPasswordProp] <> '' then
      Params.Add('password=' + Props.Values[QueryPasswordProp]);
    Connected := True;
  end;
  FQuery := TQuery.Create(nil);
  with FQuery do
  begin
    DatabaseName := FDatabase.DatabaseName;
    SQL.Text     := QuerySQL;
    AfterScroll  := QueryAfterScroll;
    Active       := True;
  end;
end;

{ Release resources }
destructor TMailTemplateQuery.Destroy;
begin
  FFields.Free;
  FQuery.Active := False;
  FQuery.Free;
  FDatabase.Connected := False;
  FDatabase.Free;
  inherited Destroy;
end;

{ Return end of file status }
function TMailTemplateQuery.GetEOF: Boolean;
begin
  Result := FQuery.EOF;
end;

{ Move to the next record }
procedure TMailTemplateQuery.NextRecord;
begin
  FQuery.Next;
end;

{ Set up the list of fields and values }
procedure TMailTemplateQuery.QueryAfterScroll(DataSet: TDataSet);
var
  Index: Integer;
begin
  with FQuery do
    for Index := 0 to FieldCount - 1 do
      if Fields[Index].DisplayText = '' then
        { If string value is empty then entry doesn't appear
          in the list, so replace it }
        FFields.Values[Fields[Index].FieldName] := Empty
      else
        FFields.Values[Fields[Index].FieldName] := Fields[Index].DisplayText;
end;

end.
